﻿using System;
using System.Threading.Tasks;
using AuthChecker;
using MediatR;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using QualityManagement.Data.Dto.Document;
using QualityManagement.Data.Dto.RiskManagement;
using QualityManagement.MediatR;
using QualityManagement.MediatR.Capa;
using QualityManagement.MediatR.RiskManagement;

namespace QualityManagement.API.Controllers.Risk_Management
{
    /// <summary>
    /// Controller for Risk Attachment
    /// </summary>
    [Route("api/[controller]")]
    [ApiController]
    [Authorize]
    public class RiskAttachmentController(IMediator _mediator) : BaseController
    {

        /// <summary>
        /// create a RiskAttachment
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPost]
        [ClaimCheck("ADD_RISK_ATTACHMENT")]
        public async Task<IActionResult> AddRiskAttachment(AddRiskAttachmentCommand command)
        {
            var result = await _mediator.Send(command);
            return GenerateResponse(result);
        }



        /// <summary>
        /// upload A RiskAttachment chunk
        /// </summary>
        /// <param name="command"></param>
        /// <returns></returns>
        [HttpPost("Chunk")]
        public async Task<IActionResult> UploadRiskAttachmentChunk([FromForm] UploadRiskAttachmentChunkCommand command)
        {
            var result = await _mediator.Send(command);
            return GenerateResponse(result);
        }



        /// <summary>
        /// Mark all chunks of a RiskAttachment  as uploaded.
        /// </summary>
        /// <param name="markRiskAttachmentChunksUploadedCommand"></param>
        /// <returns></returns>
        [HttpPost("Chunk/uploadStatus")]
        [Produces("application/json", "application/xml", Type = typeof(RiskAttachmentChunkStatus))]
        public async Task<IActionResult> MarkRiskAttachmentAllChunkUpload(MarkRiskAttachmentChunksUploadedCommand markRiskAttachmentChunksUploadedCommand)
        {
            var result = await _mediator.Send(markRiskAttachmentChunksUploadedCommand);
            return GenerateResponse(result);
        }



        /// <summary>
        /// Delete  RiskAttchament by RiskAttachment id
        /// </summary>
        /// <param name="Id"></param>
        /// <returns></returns>
        [HttpDelete("{Id}")]
        [ClaimCheck("DELETE_RISK_ATTACHMENT")]
        public async Task<IActionResult> DeleteRiskAttachment(Guid Id)
        {
            var deleteRiskAttachment = new DeleteRiskAttachmentCommand { Id = Id };
            var result = await _mediator.Send(deleteRiskAttachment);
            return GenerateResponse(result);
        }

        /// <summary>
        /// Get all RiskAttachment by Risk id
        /// </summary>
        /// <param name="id"></param>
        /// <returns></returns>
        [HttpGet("{id}")]
        [ClaimCheck("VIEW_RISK_ATTACHMENT")]
        public async Task<IActionResult> GetAllCRiskAttachment(Guid id)
        {
            var result = await _mediator.Send(new GetAllRiskAttachmentCommand() { RiskId = id });
            return GenerateResponse(result);
        }

        /// <summary>
        /// Get Download RiskAttachment chunk.
        /// </summary>
        /// <param name="riskAttachmentId"></param>
        /// <param name="chunkIndex"></param>
        /// <returns></returns>
        [HttpGet("{riskAttachmentId}/chunk/{chunkIndex}/download")]
        public async Task<IActionResult> DownloadRiskAttachment(Guid riskAttachmentId, int chunkIndex)
        {
            var result = new DownloadRiskAttachmentCommand
            {
                RiskAttachmentId = riskAttachmentId,
                ChunkIndex = chunkIndex
            };

            var downloadAttachmentResponse = await _mediator.Send(result);
            if (!downloadAttachmentResponse.Success)
            {
                return GenerateResponse(downloadAttachmentResponse);
            }

            return Ok(downloadAttachmentResponse.Data);
        }

        /// <summary>
        /// Gets the document token.
        /// </summary>
        /// <param name="id">The identifier.</param>
        /// <returns></returns>
        [HttpGet("{id}/token")]
        [AllowAnonymous]
        public async Task<IActionResult> GetDocumentToken(Guid id)
        {
            var result = await _mediator.Send(new GetDocumentTokenByDocumentIdCommand { Id = id });
            return Ok(new { result = result.Data });
        }

        /// <summary>
        /// Get Document File By Token
        /// </summary>
        /// <param name="id"></param>
        /// <param name="officeViewerRequest"></param>
        /// <returns></returns>
        [HttpGet("{id}/officeviewer")]
        [AllowAnonymous]
        public async Task<IActionResult> GetDocumentFileByToken(Guid id, [FromQuery] OfficeViewerRequest officeViewerRequest)

        {
            var command = new DownloadRiskAttachmentChunkCommand
            {
                Id = id,
                Token = officeViewerRequest.Token
            };

            var downloadDocumentResponse = await _mediator.Send(command);

            if (!downloadDocumentResponse.Success)
            {
                return GenerateResponse(downloadDocumentResponse);
            }

            var downloadDocument = downloadDocumentResponse.Data;
            return File(downloadDocument.Data, downloadDocument.ContentType, downloadDocument.FileName);
        }


    }
}
